/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.struts2.dispatcher.filter;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts2.RequestUtils;
import org.apache.struts2.StrutsStatics;
import org.apache.struts2.dispatcher.Dispatcher;
import org.apache.struts2.dispatcher.ExecuteOperations;
import org.apache.struts2.dispatcher.InitOperations;
import org.apache.struts2.dispatcher.PrepareOperations;
import org.apache.struts2.dispatcher.mapper.ActionMapping;

import java.io.IOException;

/**
 * Handles both the preparation and execution phases of the Struts dispatching process.  This filter is better to use
 * when you don't have another filter that needs access to action context information, such as Sitemesh.
 */
public class StrutsPrepareAndExecuteFilter implements StrutsStatics, Filter {

    private static final Logger LOG = LogManager.getLogger(StrutsPrepareAndExecuteFilter.class);

    protected PrepareOperations prepare;
    protected ExecuteOperations execute;

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        InitOperations init = createInitOperations();
        Dispatcher dispatcher = null;
        try {
            FilterHostConfig config = new FilterHostConfig(filterConfig);
            dispatcher = init.initDispatcher(config);
            init.initStaticContentLoader(config, dispatcher);

            prepare = createPrepareOperations(dispatcher);
            execute = createExecuteOperations(dispatcher);

            postInit(dispatcher, filterConfig);
        } finally {
            if (dispatcher != null) {
                dispatcher.cleanUpAfterInit();
            }
            init.cleanup();
        }
    }

    /**
     * Creates a new instance of {@link InitOperations} to be used during
     * initialising {@link Dispatcher}
     *
     * @return instance of {@link InitOperations}
     */
    protected InitOperations createInitOperations() {
        return new InitOperations();
    }

    /**
     * Creates a new instance of {@link PrepareOperations} to be used during
     * initialising {@link Dispatcher}
     *
     * @return instance of {@link PrepareOperations}
     */
    protected PrepareOperations createPrepareOperations(Dispatcher dispatcher) {
        return new PrepareOperations(dispatcher);
    }

    /**
     * Creates a new instance of {@link ExecuteOperations} to be used during
     * initialising {@link Dispatcher}
     *
     * @return instance of {@link ExecuteOperations}
     */
    protected ExecuteOperations createExecuteOperations(Dispatcher dispatcher) {
        return new ExecuteOperations(dispatcher);
    }

    /**
     * Callback for post initialization
     *
     * @param dispatcher   the dispatcher
     * @param filterConfig the filter config
     */
    protected void postInit(Dispatcher dispatcher, FilterConfig filterConfig) {
    }

    @Override
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException {

        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        try {
            prepare.trackRecursion(request);
            String uri = RequestUtils.getUri(request);
            if (prepare.isUrlExcluded(request)) {
                LOG.trace("Request: {} is excluded from handling by Struts, passing request to other filters", uri);
                chain.doFilter(request, response);
            } else {
                tryHandleRequest(chain, request, response, uri);
            }
        } finally {
            prepare.cleanupRequest(request);
        }
    }

    private void tryHandleRequest(FilterChain chain, HttpServletRequest request, HttpServletResponse response, String uri) throws IOException, ServletException {
        LOG.trace("Checking if: {} is a static resource", uri);
        boolean handled = execute.executeStaticResourceRequest(request, response);
        if (!handled) {
            LOG.trace("Uri: {} is not a static resource, assuming action", uri);
            handleRequest(chain, request, response, uri);
        }
    }

    private void handleRequest(FilterChain chain, HttpServletRequest request, HttpServletResponse response, String uri) throws ServletException, IOException {
        prepare.setEncodingAndLocale(request, response);
        prepare.createActionContext(request, response);
        prepare.assignDispatcherToThread();

        HttpServletRequest wrappedRequest = prepare.wrapRequest(request);
        try {
            ActionMapping mapping = prepare.findActionMapping(wrappedRequest, response, true);
            if (mapping == null) {
                LOG.trace("Cannot find mapping for: {}, passing to other filters", uri);
                chain.doFilter(request, response);
            } else {
                LOG.trace("Found mapping: {} for: {}", mapping, uri);
                execute.executeAction(wrappedRequest, response, mapping);
            }
        } finally {
            prepare.cleanupWrappedRequest(wrappedRequest);
        }
    }

    @Override
    public void destroy() {
        prepare.cleanupDispatcher();
    }

}
