/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *   * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */

package org.apache.synapse.config.xml;

import org.apache.axiom.om.OMElement;
import org.apache.synapse.Mediator;
import org.apache.synapse.mediators.eip.sample.SamplingThrottleMediator;

/**
 * Serializes the {@link org.apache.synapse.mediators.eip.sample.SamplingThrottleMediator} instance
 * into a XML configuration as follows</p>
 *
 * <pre>&lt;sampler id="string" rate="int"/&gt;</pre>
 *
 * @see org.apache.synapse.config.xml.AbstractMediatorSerializer
 */
public class SamplingThrottleMediatorSerializer extends AbstractMediatorSerializer {

    @Override
    public OMElement serializeSpecificMediator(Mediator mediator) {
        OMElement samplerElem = fac.createOMElement("sampler", synNS);
        saveTracingState(samplerElem, mediator);

        SamplingThrottleMediator samplingThrottleMediator = (SamplingThrottleMediator) mediator;

        if (samplingThrottleMediator.getId() != null) {
            samplerElem.addAttribute("id", samplingThrottleMediator.getId(), nullNS);
        }
        if (samplingThrottleMediator.getSamplingRate() != 1) {
            samplerElem.addAttribute("rate",
                    Integer.toString(samplingThrottleMediator.getSamplingRate()), nullNS);
        }
        if (samplingThrottleMediator.getUnitTime() != 1000) {
            samplerElem.addAttribute("unitTime",
                    Long.toString(samplingThrottleMediator.getUnitTime()), nullNS);
        }

        if (samplingThrottleMediator.isMessageQueueExplicitlySet()) {
            OMElement messageQueueElem = fac.createOMElement("messageQueue", synNS);
            messageQueueElem.addAttribute("class",
                    samplingThrottleMediator.getMessageQueue().getClass().getName(), nullNS);
            samplerElem.addChild(messageQueueElem);
        }

        if (samplingThrottleMediator.getTarget() != null) {
            samplerElem.addChild(
                    TargetSerializer.serializeTarget(samplingThrottleMediator.getTarget()));
        } else {
            handleException("Couldn't find the target for the sampler. " +
                    "Target is mandatory for a sampler");
        }

        return samplerElem;
    }

    @Override
    public String getMediatorClassName() {
        return SamplingThrottleMediator.class.getName();
    }
}
